/** @file   controlevent.cpp
 * @brief   Implementation of ControlEvent - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "controlevent.h"
#include "warglobals.h"
#include "objectmessage.h"
#include "MCapturable.h"
#include "MusicControlObject.h"
#include "www_assert.h"
using std::istream;
using std::string;
using std::vector;
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================

const ControlEvent::ItemType ControlEvent::CTRL_SCREENLOCK;
const ControlEvent::ItemType ControlEvent::CTRL_AI_TARGETS;
const ControlEvent::ItemType ControlEvent::CTRL_OBJECT_TELEPORT;
const ControlEvent::ItemType ControlEvent::CTRL_OBJECT_CAPTURE;
const ControlEvent::ItemType ControlEvent::CTRL_MUSIC_CONTROL;

///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
ControlEvent::ControlEvent() :
  DynamicEvent()
{
}



/** Destructor
 */
ControlEvent::~ControlEvent()
{
  for ( int i=0; i < m_controlItems.size(); i++ ) {
    delete m_controlItems.at(i);
  }
  m_controlItems.clear();
}




///
/// Public methods
/// ==============

/** Activates this event
 */
void ControlEvent::activate()
{
  this->status( DynamicEvent::ACTIVE );
}



/** Updates this event
 */
void ControlEvent::update()
{
  if ( this->status() == DynamicEvent::ACTIVE ) {
    DynamicEvent::m_counter -= 1;
    if ( DynamicEvent::m_counter < 0 ) {
      this->launchEvent();
      this->status( DynamicEvent::COMPLETE );
    }
  }
}



/** Reads the contents
 */
int ControlEvent::readData( istream& rIn )
{
  LOG_MESSAGE( "Reading Control-event" );
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</control>" ) {
      break;
      
    } else if ( tmp == "<screen_lock>" ) {
      ControlItem* pI = this->readScreenLock( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "<ai_targets>" ) {
      ControlItem* pI = this->readAITargets( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "<object_message>" ) {
      ControlItem* pI = this->readObjectMessage( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "<object_teleport>" ) {
      ControlItem* pI = this->readObjectTeleport( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "<object_capture>" ) {
      ControlItem* pI = this->readObjectCapture( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "<music_control>" ) {
      ControlItem* pI = this->readMusicControl( rIn );
      if ( pI == 0 ) {
        return -1;
      }
      m_controlItems.push_back( pI );
      
    } else if ( tmp == "delay:" ) {
      rIn >> DynamicEvent::m_counter;
      
    } else {
      int ret = alert( "Unsupported Control element", tmp.c_str(), 0,
                       "quit", "continue", 0,0 );
      if ( ret == 1 ) {
        return -1;
      }
      
    }
  }
  return 0;
}




///
/// Getter methods
/// ==============




///
/// Private or Protected methods
/// ============================

/** Reads the screenlock element
 */
ControlEvent::ControlItem* ControlEvent::readScreenLock( istream& rIn )
{
  ControlItem* pI = new ControlItem;
  pI->type = CTRL_SCREENLOCK;
  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered while parsing the", "SCREEN_LOCK", 0,
             "Quit",0, 0,0 );
      delete pI;
      return 0;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</screen_lock>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "status:" ) {
      rIn >> tmp;
      if ( tmp == "on" ) {
        pI->intData1 = 1;
      } else {
        pI->intData1 = 0;
      }
      
    } else if ( tmp == "topleft:" ) {
      rIn >> pI->pos1.vx >> pI->pos1.vy;
      pI->pos1 *= 32;
      Vec2D p2( pI->pos1 );
      p2 += Vec2D( 640,480 );
      p2 /= 2;
      pI->pos2 = p2;
      
    } else if ( tmp == "player_pos:" ) {
      rIn >> pI->pos2.vx >> pI->pos2.vy;
      pI->pos2 *= 32;
      
    } else {
      int ret = alert( "Unsuported CONTROL_EVENT - parameter", tmp.c_str(), 0,
                       "Quit", "Continue", 0,0 );
      if ( ret == 1 ) {
        delete pI;
        return 0;
      }
    }
  }
  return pI;
}



/** Reads the data of ai-target
 */
ControlEvent::ControlItem* ControlEvent::readAITargets( istream& rIn )
{
  ControlItem* pC = new ControlItem;
  pC->type = CTRL_AI_TARGETS;
  pC->intData1 = -1;
  while ( true ) {
    if ( rIn.eof() == true ) {
      delete pC;
      alert( "EOF encountered while parsing the", "AI_TARGETS - element", 0,
             "Quit", 0, 0,0 );
      return 0;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</ai_targets>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "primary_target_id:" ) {
      rIn >> pC->intData1;
      
    } else {
      int ret = alert( "Unsuported CONTROL_EVENT - parameter", tmp.c_str(), 0,
                       "Quit", "Continue", 0,0 );
      if ( ret == 1 ) {
        delete pC;
        return 0;
      }
    }
  }
  return pC;
}



/** Reads the controller - message
 */
ControlEvent::ControlItem* ControlEvent::readObjectMessage( istream& rIn )
{
  ControlItem* pC = new ControlItem;
  pC->type = CTRL_OBJECT_MESSAGE;
  pC->intData1 = -1;
  pC->intData2 = 0;
  while ( true ) {
    if ( rIn.eof() == true ) {
      delete pC;
      alert( "EOF encountered while parsing the", "CONTROLLER_MESSAGE - element",
             0, "Quit",0, 0,0 );
      return 0;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "</object_message>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "object_id:" ) {
      rIn >> pC->idcode;
      
    } else if ( tmp == "message_id:" ) {
      rIn >> pC->intData1;
      
    } else if ( tmp == "message_data:" ) {
      rIn >> pC->intData2;
      
    } else {
      int ret = alert( "Unsupported CONTROL_EVEN - parameter", tmp.c_str(),0,
                       "Quit", "Continue", 0,0 );
      if ( ret == 1 ) {
        delete pC;
        return 0;
      }
    }
  }
  return pC;
}



/** Reads the object teleport element
 */
ControlEvent::ControlItem* ControlEvent::readObjectTeleport( istream& rIn )
{
  ControlItem* pC = new ControlItem;
  pC->type = CTRL_OBJECT_TELEPORT;
  pC->idcode = -1;
  pC->pos1 = Vec2D(0,0);
  pC->intData1 = 0;
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      delete pC;
      alert( "EOF encountered while parsing the", "OBJECT_TELEPORT - element",
             0, "Quit",0, 0,0 );
      return 0;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</object_teleport>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "object_id:" ) {
      rIn >> pC->idcode;
      
    } else if ( tmp == "target_pos:" ) {
      rIn >> pC->pos1.vx >> pC->pos1.vy >> pC->intData1;
      pC->pos1 *= 32;
      
    } else if ( tmp == "target_offset:" ) {
      rIn >> pC->pos2.vx >> pC->pos2.vy;
      
    } else {
      int ret = alert( "Unsupported OBJECT_TELEPORT - parameter", tmp.c_str(),0,
                       "Quit", "Continue", 0,0 );
      if ( ret == 1 ) {
        delete pC;
        return 0;
      }
    }
  }
  return pC;
}



/** Reads the object capture data
 */
ControlEvent::ControlItem* ControlEvent::readObjectCapture( istream& rIn )
{
  ControlItem* pC = new ControlItem;
  pC->type = CTRL_OBJECT_CAPTURE;
  pC->idcode = -1;
  pC->pos1 = Vec2D(0,0);
  pC->intData1 = -1;
  pC->intData2 = -1;
  pC->intData3 = -1;

  while ( true ) {
    if ( rIn.eof() == true ) {
      delete pC;
      alert( "EOF encountered while parsing the", "OBJECT_CAPTURE - element",
             0, "Quit",0, 0,0 );
      return 0;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "</object_capture>" ) {
      break;

    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "object_id:" ) {
      rIn >> pC->idcode;

    } else if ( tmp == "capture_id:" ) {
      rIn >> pC->intData1;
      
    } else if ( tmp == "no_exit:" ) {
      rIn >> pC->intData2;

    } else {
      int ret = alert( "Unsupported OBJECT_CAPTURE - parameter", tmp.c_str(),0,
                       "Quit", "Continue", 0,0 );
      if ( ret == 1 ) {
        delete pC;
        return 0;
      }
    }
  }
  return pC;
}

ControlEvent::ControlItem* ControlEvent::readMusicControl( istream& aIn )
{
  ControlItem* item = new ControlItem;
  item->type = CTRL_MUSIC_CONTROL;
  item->intData1 = -1;
  item->intData2 = -1;
  item->intData3 = -1;
  while ( true ) {
    if ( aIn.eof() == true ) {
      delete item;
      return 0;
    }
    string tmp;
    aIn >> tmp;
    if ( tmp == "#" ) {
      aIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</music_control>") {
      break;
      
    } else if ( tmp == "action:" ) {
      aIn >> tmp;
      if ( tmp == "fade_out" ) {
        item->intData1 = EFadeOut;
      } else if ( tmp == "fade_in" ) {
        item->intData1 = EFadeIn;
      } else if ( tmp == "set_volume" ) {
        item->intData1 = ESetVolume;
      } else if ( tmp == "change_music" ) {
        item->intData1 = EChangeMusic;
      } else if ( tmp == "stop_playback" ) {
        item->intData1 = EStopMusic;
      }
      
    } else if ( tmp == "to_volume:" || tmp == "volume:" || tmp == "track_number:" ) {
      aIn >> item->intData2;
      
    } else if ( tmp == "fade_speed:" ) {
      aIn >> item->intData3;
      
    }
  }
  return item;
}



/** Launches the events
 */
void ControlEvent::launchEvent()
{
  for ( int i=0; i < m_controlItems.size(); i++ ) {
    ControlItem* pI = m_controlItems.at(i);
    switch ( pI->type ) {
      case ( CTRL_SCREENLOCK ): {
        this->doScreenLock( pI );
        break;
        
      }
      case ( CTRL_AI_TARGETS ): {
        this->doAiTargets( pI );
        break;
        
      }
      case ( CTRL_OBJECT_MESSAGE ): {
        this->doObjectMessage( pI );
        break;
      }
      case ( CTRL_OBJECT_TELEPORT ): {
        this->doObjectTeleport( pI );
        break;
      }
      case ( CTRL_OBJECT_CAPTURE ): {
        this->doObjectCapture( pI );
        break;
      }
      case ( CTRL_MUSIC_CONTROL ): {
        this->doMusicControl( pI );
        break;
      }
    }
  }
}



/** Locks the screen or releases it.
 */
void ControlEvent::doScreenLock( ControlItem* pI )
{
  if ( pI->intData1 == 1 ) {
    WarGlobals::screenLockMode = true;
    WarGlobals::screenLockPos = pI->pos1;
    if ( pI->pos2.intX() != -1 && pI->pos2.intY() != -1 ) {
      WarGlobals::pObjTable->pPlayer->position( pI->pos2 );
    }
    
  } else {
    WarGlobals::screenLockMode = false;
    
  }
}



/** Changes the primary ai target
 */
void ControlEvent::doAiTargets( ControlItem* pI )
{
  if ( pI->intData1 == -1 ) {
    WarGlobals::pObjTable->pPrimaryTarget = WarGlobals::pObjTable->pPlayer;
    
  } else {
    vector< GameObject* > objVec;
    bool b = WarGlobals::pObjTable->getObjectsWithID( pI->intData1, objVec );
    if ( objVec.size() > 0 ) {
      WarGlobals::pObjTable->pPrimaryTarget = objVec.at(0);
    }
  }
}



/** Sends a message to GameObject
 */
void ControlEvent::doObjectMessage( ControlItem* pI )
{
  if ( pI->idcode == -1 ) {
    // Sends the same message to every gameobject. NOT IMPLEMENTED !
    return;
  }
  
  // Construct the message.
  ObjectMessage tmpMessage;
  tmpMessage.m_id = pI->intData1;
  
  char buff[10];
  string paramName = "message_data:";
  string paramValue = string( itoa( pI->intData2, buff, 10 ) );
  tmpMessage.m_params.addParameter( paramName, paramValue );
  
  // Send the message to the GameObjects that match with the id-code.
  vector<GameObject*> objs;
  WarGlobals::pObjTable->getObjectsWithID( pI->idcode, objs );
  for ( int i=0; i < objs.size(); i++ ) {
    objs.at(i)->messagePort( tmpMessage );
  }
}



/** Teleports the object.
 */
void ControlEvent::doObjectTeleport( ControlItem* pI )
{
  if ( pI->idcode == -1 ) {
    return;
  }
  
  Vec2D pos = pI->pos1 + pI->pos2;
  
  vector< GameObject* > objects;
  WarGlobals::pObjTable->getObjectsWithID( pI->idcode, objects );
  
  for ( int i=0; i < objects.size(); i++ ) {
    objects.at(i)->position( pos );
    objects.at(i)->angle( pI->intData1 );
  }
}



/** Captures another object.
 */
void ControlEvent::doObjectCapture( ControlItem* pI )
{
  if ( pI->idcode == -1 || pI->intData1 == -1 ) {
    return;
  }

  int idcode1 = pI->idcode;
  int idcode2 = pI->intData1;
  vector< GameObject* > masterObjects;
  vector< GameObject* > captureObjects;
  
  bool found = true;
  found &= WarGlobals::pObjTable->getObjectsWithID(idcode1, masterObjects);
  found &= WarGlobals::pObjTable->getObjectsWithID(idcode2, captureObjects);
  
  if ( found == false ) {
    return;
  }
  
  
  GameObject* master = masterObjects.at(0);
  MCapturable* target = dynamic_cast<MCapturable*>( captureObjects.at(0) );
  if ( master == 0 || target == 0 ) {
    return;
  }
  
  if ( target->canBeCaptured() == true ) {
    target->capture( master );
    WarGlobals::captureFeatureOn = !static_cast<bool>( pI->intData2 );
    
  } else {
    if ( target->getMaster() != 0 ) {
      // The target object is already captured. Release it.
      target->capture( 0 );
      WarGlobals::captureFeatureOn = true;
    }
  }
}

void ControlEvent::doMusicControl( ControlItem* pI )
{
  vector< GameObject* > objects;
  bool found = WarGlobals::pObjTable->getObjectsWithID( KMp3PlayerControlObjectId, objects );
  if ( found == false ) {
    return;
  }
  MusicControlObject* obj = dynamic_cast<MusicControlObject*>( objects.at(0) );
  switch ( pI->intData1 ) {
    case (EFadeOut):
    case (EFadeIn): {
      obj->launchVolumeFade( pI->intData2, pI->intData3 );
      break;
    }
    case (ESetVolume): {
      obj->launchSetVolume( pI->intData2 );
      break;
    }
    case (EChangeMusic): {
      obj->launchChangeSong( pI->intData2 );
      break;
    }
    case (EStopMusic): {
      obj->launchStopSong();
      break;
    }
  }
}


}  // end of namespace
